<?php
/**
 * @author Jegtheme
 */

namespace JNEWS_VIEW_COUNTER;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class JNews View Counter Init
 */
class Init {
	/**
	 * @var Init
	 */
	private static $instance;

	/**
	 * @var Counter
	 */
	public $counter;

	/**
	 * @var Crawler_Detect
	 */
	public $crawler_detect;

	/**
	 * @var Frontend
	 */
	public $frontend;

	/**
	 * @var Template
	 */
	public $template;

	/**
	 * @var Options\Options
	 */
	public $settings;

	/**
	 * @var Dashboard
	 */
	public $dashboard;

	/**
	 * View counter options
	 *
	 * @var array
	 */
	public $options;

	/**
	 * View counter options
	 *
	 * @var array
	 */
	public $defaults = array(
		'general' => array(
			'strict_counts'       => false,
			'time_between_counts' => array(
				'number' => 0,
				'type'   => 'hours',
			),
			'exclude'             => array(
				'groups' => array( 'robots' ),
				'roles'  => array(),
			),
			'log'                 => array(
				'limit'         => false,
				'expires_after' => 180,
			),
			'sync_multilang'      => true,
		),
		'display' => array(),
		'config'  => array(
			'dates'         => '',
			'range'         => 'last7days',
			'time_quantity' => 24,
			'time_unit'     => 'hours',
			'limit'         => 10,
			'freshness'     => false,
			'post_type'     => 'post',
			'cpt'           => array(), //see qMcFaETm
		),
		'cache'   => array(
			'total_view'      => true,
			'tv_expired_time' => 0,
			'tv_expired_unit' => '60',
			'popular_query'   => true,
			'pq_expired_time' => 0,
			'pq_expired_unit' => '60',

		),
	);

	/**
	 * Disable object cloning.
	 */
	public function __clone() {}

	/**
	 * Disable unserializing of the class.
	 */
	public function __wakeup() {}

	/**
	 * Instance of Init JNews View Counter
	 *
	 * @return Init
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Init ) ) {
			self::$instance = new Init();
			self::$instance->includes();
			self::$instance->counter        = new Counter();
			self::$instance->crawler_detect = new Crawler_Detect();
			self::$instance->frontend       = new Frontend();
			self::$instance->template       = new Template();
			self::$instance->settings       = new Options\Options();
			if ( is_admin() || ( isset( $_SERVER['REQUEST_URI'] ) && strpos( $_SERVER['REQUEST_URI'], 'wp-cron.php' ) !== false ) ) {
				self::$instance->dashboard = new Dashboard();
			}
		}
		return self::$instance;
	}

	public function includes() {
		require_once JNEWS_VIEW_COUNTER_DIR . 'class/functions.php';
	}

	/**
	 * Construct of JNews View Counter
	 */
	private function __construct() {
		$this->options = array(
			'general' => array_merge( $this->defaults['general'], Helper::get_view_counter_option( 'general', $this->defaults['general'] ) ),
			'display' => array_merge( $this->defaults['display'], Helper::get_view_counter_option( 'display', $this->defaults['display'] ) ),
			'config'  => array_merge( $this->defaults['config'], Helper::get_view_counter_option( 'config', $this->defaults['config'] ) ),
			'cache'   => array_merge( $this->defaults['cache'], Helper::get_view_counter_option( 'cache', $this->defaults['cache'] ) ),
		);
		if ( ! empty( $this->options['config']['cpt'] ) ) { //see qMcFaETm
			$post_type                            = array_merge( array( 'post' ), $this->options['config']['cpt'] );
			$this->options['config']['post_type'] = implode( ',', $post_type );
		}
		wp_cache_add_global_groups(
			array(
				'jnews-view-counter',
			)
		); /* see XjvoYQZ7 */
		register_activation_hook( JNEWS_VIEW_COUNTER_FILE, array( 'JNEWS_VIEW_COUNTER\Activator', 'activate' ) );
		register_deactivation_hook( JNEWS_VIEW_COUNTER_FILE, array( 'JNEWS_VIEW_COUNTER\Deactivator', 'deactivate' ) );
		$this->setup_hook();
		$this->load_plugin_text_domain();
	}

	private function setup_hook() {
		add_action( 'wp_enqueue_scripts', array( 'JNEWS_VIEW_COUNTER\Helper', 'wp_localize_vanillajs_datepicker' ), 1000 );
		add_action( 'admin_enqueue_scripts', array( 'JNEWS_VIEW_COUNTER\Helper', 'wp_localize_vanillajs_datepicker' ), 1000 );
		add_filter( 'posts_clauses', array( $this, 'sort_post_by_view' ), 10, 2 );
		add_filter( 'jnews_view_counter_query_join', array( $this, 'multi_lang_query_join' ), 99, 2 );
		add_filter( 'jnews_view_counter_query_where', array( $this, 'multi_lang_query_where' ), 99, 2 );
	}

	/**
	 * Load plugin text domain
	 */
	private function load_plugin_text_domain() {
		load_plugin_textdomain( JNEWS_VIEW_COUNTER, false, basename( JNEWS_VIEW_COUNTER_DIR ) . '/languages/' );
	}

	/**
	 * Sorting Post by Post View
	 */
	public function sort_post_by_view( $clauses, $query ) {
		if ( ! is_admin() || ! $query->is_main_query() ) {
			return $clauses;
		}
		global $wpdb;

		if ( $query->get( 'post_type' ) === 'post' && $query->get( 'orderby' ) === 'view_count' ) {
			$clauses['join']   .= "LEFT JOIN {$wpdb->prefix}popularpostsdata ON {$wpdb->prefix}posts.ID = {$wpdb->prefix}popularpostsdata.postid";
			$clauses['orderby'] = "{$wpdb->prefix}popularpostsdata.pageviews " . $query->get( 'order' );
		}

		return $clauses;
	}

	/**
	 * Adding multi language filter on view counter JOIN Query.
	 *
	 * @param string $join JNews View Counter JOIN QUERY.
	 * @param object $instance JNews Element Instance.
	 */
	public function multi_lang_query_join( $join, $instance ) {
		if ( ! JNews_View_Counter()->options['general']['sync_multilang'] ) {
			if ( function_exists( 'pll_get_post' ) ) {
				global $wpdb;
				$join .= "
				INNER JOIN `{$wpdb->prefix}term_relationships` tr ON p.ID = tr.object_id
				INNER JOIN `{$wpdb->prefix}term_taxonomy` tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
				INNER JOIN `{$wpdb->prefix}terms` t ON tt.term_id = t.term_id
			";

			} elseif ( defined( 'ICL_LANGUAGE_CODE' ) ) {
				global $wpdb;
				$join .= "
				JOIN `{$wpdb->prefix}icl_translations` tr
				ON p.ID = tr.element_id
				AND tr.element_type = CONCAT('post_', p.post_type)
			";
			}
		}
		return $join;
	}

	/**
	 * Adding multi language filter on view counter WHERE Query.
	 *
	 * @param string $where JNews View Counter WHERE QUERY.
	 * @param object $instance JNews Element Instance.
	 */
	public function multi_lang_query_where( $where, $instance ) {
		if ( ! JNews_View_Counter()->options['general']['sync_multilang'] ) {
			if ( function_exists( 'pll_current_language' ) ) {
				$where = $where . '
			AND tt.taxonomy = "language"
			AND t.slug ="' . pll_current_language() . '"';
			} elseif ( defined( 'ICL_LANGUAGE_CODE' ) ) {
				$where .= "
				AND tr.language_code = '" . ICL_LANGUAGE_CODE . "'";
			}
		}
		return $where;
	}
}
